// This script exports all currently open InDesign documents to
// the user's choice of PDF preset. Presets are read from a text file
// called "PDF_presets.txt" located in the same directory as the 
// script. This is most easily created from a directory listing
// of the PDF .joboptions files, located in (Windows)
// appdata\roaming\Adobe\Adobe PDF\Settings.
//
// Each PDF is exported to the same directory as its InDesign file, 
// with the same filename as the INDD.
//
// EXISTING PDFs WILL BE OVERWRITTEN WITHOUT WARNING if they have the
// same name.
//
// I created this script with the heavy lifting being done by
// Windows 11 Copilot to something I find myself doing several times
// a day on a current long publication project, whereby multiple
// documents in a set need to be exported to individual
// PDFs for review by different people.
//
// A bonus for those of us with a long set of presets is that the
// dropdown presents them in the order they appear in the text
// file, not the order in which InDesign would normally list them.
// This means you can keep the most-used presets at the top of the
// list. 
//
// Alan Gilbertson
// 17 March 2024
// *****************************************

// Function reads the PDF preset names from a PDF_presets.txt,
// which must be in the same directory as the script.
function readPresetsFromFile() {
  var scriptFile = new File($.fileName);
  var presetsFile = new File(scriptFile.path + "/pdf_presets.txt");
  var presets = [];
  if (presetsFile.exists) {
    presetsFile.open('r');
    while (!presetsFile.eof) {
      var line = presetsFile.readln();
      if (line) presets.push(line);
    }
    presetsFile.close();
    return presets;
  } else {
    alert("The PDF presets file must exist in the same folder as this script.");
    return null;
  }
}

// Function creates a dropdown list and asks which preset to use
function choosePreset(presets) {
  var w = new Window('dialog', 'Select PDF Export Preset');
  var dropdown = w.add('dropdownlist', undefined, presets);
  dropdown.selection = 0;
  w.add('button', undefined, 'OK', {name: 'ok'});
  w.add('button', undefined, 'Cancel', {name: 'cancel'});

  if (w.show() == 1) {
    return dropdown.selection.text;
  } else {
    return null;
  }
}

// Function exports each document to PDF using the chosen preset
function exportToPDF(doc, presetName) {
  var myPDFExportPreset = app.pdfExportPresets.itemByName(presetName);
  if (!myPDFExportPreset.isValid) {
    alert("The PDF export preset '" + presetName + "' does not exist.");
    return;
  }
  var pdfFile = new File(doc.filePath + "/" + doc.name.replace(/\.indd$/, ".pdf"));
  doc.exportFile(ExportFormat.PDF_TYPE, pdfFile, false, myPDFExportPreset);
}

// Main script execution
if (app.documents.length === 0) {
  alert("No documents are open. Please open a document and try again.");
} else {
  var presetsList = readPresetsFromFile();
  if (presetsList) {
    var selectedPreset = choosePreset(presetsList);
    if (selectedPreset) {
      for (var i = 0; i < app.documents.length; i++) {
        var currentDoc = app.documents[i];
        exportToPDF(currentDoc, selectedPreset);
      }
	  alert("All documents have been successfully exported to PDF.");
    } else {
      alert("No preset was selected. The export process has been canceled.");
    }
  }
}